/*
  Prof. Maffucci Michele
  25.02.26
  ESP32-C3 Super Mini — Web Server per accendere/spegnere il LED via browser

  NOTE IMPORTANTI (ESP32-C3 Super Mini):
  - Il LED integrato su molte board ESP32-C3 Super Mini è su GPIO 8 ed è in logica invertita (active LOW):
      * LOW  = LED acceso
      * HIGH = LED spento
  - In questo sketch pilotiamo il pin 8 in logica invertita e rendiamo coerenti:
      1) comando /H = ACCENDI (porta il pin a LOW)
      2) comando /L = SPEGNI (porta il pin a HIGH)
      3) badge: pallino verde quando LED è ACCESO, grigio quando è SPENTO
*/

#include <WiFi.h>

// === Credenziali Wi-Fi (modifica qui) ===
const char* nomeReteWiFi = "SOSTITUISCI_CON_IL_TUO_SSID";
const char* passwordWiFi = "SOSTITUISCI_CON_LA_TUA_PASSWORD";

// === Server HTTP sulla porta 80 ===
WiFiServer server(80);

// === Pin LED (qui usiamo GPIO 8) ===
const int pinLed = 8;

// -----------------------------------------------------------------------------
// Funzione di utilità: imposta lo stato del LED tenendo conto della logica invertita
// -----------------------------------------------------------------------------
void impostaLed(bool acceso) {
  if (acceso) {
    digitalWrite(pinLed, LOW);  // accendi LED
  } else {
    digitalWrite(pinLed, HIGH);  // spegni LED
  }
}

// -----------------------------------------------------------------------------
// Funzione di utilità: invia al browser una pagina HTML completa
// -----------------------------------------------------------------------------
void inviaPaginaWeb(WiFiClient& client, bool ledAcceso) {
  // Header HTTP
  client.println("HTTP/1.1 200 OK");
  client.println("Content-type:text/html; charset=utf-8");
  client.println("Cache-Control: no-store");
  client.println("Connection: close");
  client.println();

  // Corpo HTML
  client.print(R"rawliteral(
<!doctype html>
<html lang="it">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>ESP32-C3 Super Mini Web Server</title>
  <style>
    :root{
      --bg1:#f7f9fc;
      --bg2:#eef2ff;
      --card:#ffffff;
      --text:#0f172a;
      --muted:#475569;
      --line:#e2e8f0;
      --shadow:0 10px 30px rgba(15,23,42,.10);
      --radius:18px;
      --ok:#16a34a;
      --off:#64748b;
      --btn:#2563eb;
      --btn2:#0f172a;
    }

    *{box-sizing:border-box}
    body{
      margin:0;
      min-height:100vh;
      display:flex;
      align-items:center;
      justify-content:center;
      padding:18px;
      font-family: ui-sans-serif, system-ui, -apple-system, "Segoe UI", Roboto, Helvetica, Arial;
      color:var(--text);
      background:
        radial-gradient(900px 500px at 20% 10%, rgba(37,99,235,.15), transparent 60%),
        radial-gradient(900px 520px at 80% 15%, rgba(99,102,241,.12), transparent 60%),
        linear-gradient(180deg, var(--bg1), var(--bg2));
    }

    .card{
      width:min(720px, 100%);
      background:var(--card);
      border:1px solid var(--line);
      border-radius:var(--radius);
      box-shadow:var(--shadow);
      overflow:hidden;
    }

    header{
      padding:18px 18px 10px;
      border-bottom:1px solid var(--line);
      display:flex;
      justify-content:space-between;
      gap:14px;
      flex-wrap:wrap;
    }

    h1{
      margin:0;
      font-size:18px;
      font-weight:800;
      letter-spacing:.2px;
    }
    .sub{
      margin:6px 0 0;
      color:var(--muted);
      font-size:13px;
      line-height:1.35;
    }

    .badge{
      display:inline-flex;
      align-items:center;
      gap:10px;
      padding:10px 12px;
      border:1px solid var(--line);
      border-radius:999px;
      font-weight:800;
      font-size:13px;
      color:var(--text);
      background:#f8fafc;
      height: fit-content;
    }
    .dot{
      width:10px;height:10px;border-radius:50%;
      background: var(--off);
      box-shadow: 0 0 0 4px rgba(100,116,139,.12);
    }
    .dot.on{
      background: var(--ok);
      box-shadow: 0 0 0 4px rgba(22,163,74,.14);
    }

    main{ padding:18px; display:grid; gap:14px; }

    .row{
      display:flex;
      gap:12px;
      flex-wrap:wrap;
      align-items:center;
      justify-content:space-between;
      padding:14px;
      border:1px dashed var(--line);
      border-radius:14px;
      background:#fbfdff;
    }

    .k{ color:var(--muted); font-size:13px; }
    .v{ font-weight:800; font-size:15px; }

    .mono{
      font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
      font-size:12px;
      padding:2px 10px;
      border:1px solid var(--line);
      border-radius:999px;
      background:#f8fafc;
      color:#0f172a;
    }

    .btns{
      display:flex;
      gap:10px;
      flex-wrap:wrap;
    }

    a{ text-decoration:none; }
    .btn{
      border:0;
      border-radius:14px;
      padding:12px 14px;
      font-size:15px;
      font-weight:850;
      cursor:pointer;
      transition: transform .06s ease, filter .15s ease;
      min-width: 160px;
    }
    .btn:active{ transform: translateY(1px); }
    .btnOn{ background: var(--btn); color:#fff; }
    .btnOff{ background: #e2e8f0; color: var(--btn2); }

    footer{
      padding:14px 18px 18px;
      color:var(--muted);
      font-size:12px;
      line-height:1.45;
      border-top:1px solid var(--line);
      background:#ffffff;
    }
  </style>
</head>
<body>
  <div class="card">
    <header>
      <div>
        <h1>Esp32-C3 Super Mini Web Server</h1>
        <div class="sub">
          Controllo LED via browser: usa i pulsanti per inviare richieste HTTP <span class="mono">GET /H</span> e <span class="mono">GET /L</span>.
        </div>
      </div>

      <div class="badge">
        <span class="dot )rawliteral");

  // Inseriamo classe "on" nel pallino quando il LED è ACCESO (verde),
  // altrimenti resta grigio.
  if (ledAcceso) {
    client.print("on");
  } else {
    client.print("");
  }
  client.print(R"rawliteral("></span>
        Stato LED: )rawliteral");

  // Testo coerente con lo stato reale
  if (ledAcceso) {
    client.print("ACCESO");
  } else {
    client.print("SPENTO");
  }

  client.print(R"rawliteral(
      </div>
    </header>

    <main>
      <div class="row">
        <div>
          <div class="k">Uscita controllata</div>
          <div class="v">LED su GPIO 8</div>
        </div>
        <div class="mono">PIN 8</div>
      </div>

      <div class="row">
        <div>
          <div class="k">Comandi</div>
          <div class="v">Accendi / Spegni</div>
        </div>
        <div class="btns">
          <a href="/H"><button class="btn btnOn">ACCENDI (ON)</button></a>
          <a href="/L"><button class="btn btnOff">SPEGNI (OFF)</button></a>
        </div>
      </div>
    </main>

    <footer>
      Prof. Maffucci Michele<br>
      Modifica il codice come meglio ritenete e fatene buon uso ;-)
    </footer>
  </div>
</body>
</html>
)rawliteral");

  // Fine risposta HTTP
  client.println();
}

// -----------------------------------------------------------------------------
// setup(): inizializzazione di seriale, GPIO e Wi-Fi
// -----------------------------------------------------------------------------
void setup() {
  Serial.begin(115200);

  // Impostiamo il pin del LED come uscita digitale
  pinMode(pinLed, OUTPUT);

  // Stato iniziale: LED spento (logica invertita => HIGH = OFF)
  impostaLed(false);

  delay(10);

  // --- Connessione alla rete Wi-Fi ---
  Serial.println();
  Serial.print("Connessione alla rete Wi-Fi: ");
  Serial.println(nomeReteWiFi);

  WiFi.begin(nomeReteWiFi, passwordWiFi);

  // Attesa finché l'ESP32 non è connesso
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println();
  Serial.println("Wi-Fi connesso!");
  Serial.print("Indirizzo IP: ");
  Serial.println(WiFi.localIP());

  // Avvio server HTTP
  server.begin();
  Serial.println("Server HTTP avviato (porta 80).");
  Serial.println("Apri il browser e visita l'indirizzo IP indicato sopra.");
}

// -----------------------------------------------------------------------------
// loop(): attende connessioni client e gestisce richieste HTTP
// -----------------------------------------------------------------------------
void loop() {
  // Ascolta eventuali client in arrivo (browser)
  WiFiClient client = server.available();

  // Se non c'è un client, non facciamo nulla
  if (!client) return;

  Serial.println("Nuovo client connesso.");

  // Memorizza la riga corrente della richiesta HTTP (terminata da \r\n)
  String rigaCorrente = "";

  // Variabile per capire se il LED è acceso o spento (da mostrare nella pagina)
  bool ledAcceso = false;

  // Leggiamo i dati finché il client resta connesso
  while (client.connected()) {
    if (!client.available()) continue;

    // Legge un carattere della richiesta
    char c = client.read();

    // Stampiamo sul Serial Monitor la richiesta (utile per studio)
    Serial.write(c);

    // Se arriva il newline, abbiamo finito una riga
    if (c == '\n') {
      // Riga vuota => fine degli header HTTP, possiamo rispondere
      if (rigaCorrente.length() == 0) {

        // Con logica invertita: LOW = acceso, HIGH = spento
        ledAcceso = (digitalRead(pinLed) == LOW);

        // Invia la pagina HTML
        inviaPaginaWeb(client, ledAcceso);

        // Fine: usciamo dal ciclo di lettura
        break;
      } else {
        // Abbiamo concluso una riga non vuota: resettiamo per la prossima
        rigaCorrente = "";
      }
    } else if (c != '\r') {
      // Carattere normale: lo aggiungiamo alla riga corrente
      rigaCorrente += c;
    }

    // --- Interpretazione semplice della richiesta ---
    // Se il browser chiede "GET /H", accendiamo il LED
    if (rigaCorrente.endsWith("GET /H")) {
      impostaLed(true);  // ON => LOW
      Serial.println("\nComando ricevuto: ACCENDI LED (GET /H)");
    }

    // Se il browser chiede "GET /L", spegniamo il LED
    if (rigaCorrente.endsWith("GET /L")) {
      impostaLed(false);  // OFF => HIGH
      Serial.println("\nComando ricevuto: SPEGNI LED (GET /L)");
    }
  }

  // Chiudiamo la connessione
  client.stop();
  Serial.println("Client disconnesso.");
}